/**
* \file: WaylandTouchListener.cpp
*
* \version: 0.3
*
* \release: $Name:$
*
* Handles touch events coming from Wayland
*
* \component: Unified SPI
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <string>
#include <uspi_macros.h>
#include <uspi/ITouchFacade.h>
#include "WaylandTouchListener.h"

namespace adit { namespace uspi {
using std::to_string;

struct wl_touch_listener WaylandTouchListener::mWlTouchListener =
{
    WaylandTouchListener::onDown,
    WaylandTouchListener::onUp,
    WaylandTouchListener::onMotion,
    WaylandTouchListener::onFrame,
    WaylandTouchListener::onCancel,
    NULL,
    NULL
};

WaylandTouchListener::WaylandTouchListener(ITouchFacadeCallbacks& inCallbacks, int inDisplayWidth, int inDisplayHeight,
        int inFingerNum)
: mCallbacks(inCallbacks), mDisplayWidth(inDisplayWidth), mDisplayHeight(inDisplayHeight), mMaxFingerNum(inFingerNum) { }

WaylandTouchListener::~WaylandTouchListener() { }

void WaylandTouchListener::setLateResolution(int inWidth, int inHeight)
{
    mDisplayWidth = inWidth;
    mDisplayHeight = inHeight;
}

/* Wayland touch event handlers */
void WaylandTouchListener::onDown(void* inMe, wl_touch* inTouch, uint32_t inEventSerial, uint32_t inTimestamp,
        wl_surface* inWlSurface, int32_t inFingerId, wl_fixed_t inPixelX, wl_fixed_t inPixelY)
{
    uspi_unused_variable(inTouch);
    uspi_unused_variable(inEventSerial);
    uspi_unused_variable(inTimestamp);
    uspi_unused_variable(inWlSurface);

    auto me = static_cast<WaylandTouchListener*>(inMe);

    me->mCallbacks.onLogging(USPI_LOG_VERBOSE, "Touch down event finger: "
            + to_string(inFingerId) + "/" + to_string(me->mMaxFingerNum) + " on "
            + to_string(wl_fixed_to_double(inPixelX)) + " - " + to_string(wl_fixed_to_double(inPixelY)));

    if (inFingerId < me->mMaxFingerNum)
    {
        TouchEvent currentTouchEvent;
        currentTouchEvent.fingerId = inFingerId;
        /* Wayland does not detect pressure */
        currentTouchEvent.pressure = 1;
        currentTouchEvent.xRelative = (float)wl_fixed_to_double(inPixelX) / me->mDisplayWidth;
        currentTouchEvent.yRelative = (float)wl_fixed_to_double(inPixelY) / me->mDisplayHeight;
        currentTouchEvent.eventType = TouchEvent::down;

        me->mCallbacks.onTouch(currentTouchEvent);
    }
}

void WaylandTouchListener::onUp(void* inMe, wl_touch* inTouch,
        uint32_t inEventSerial, uint32_t inTimestamp, int32_t inFingerId)
{
    uspi_unused_variable(inTouch);
    uspi_unused_variable(inEventSerial);
    uspi_unused_variable(inTimestamp);

    auto me = static_cast<WaylandTouchListener*>(inMe);

    me->mCallbacks.onLogging(USPI_LOG_VERBOSE, "Touch up event finger: "
            + to_string(inFingerId) + "/" + to_string(me->mMaxFingerNum));

    if (inFingerId < me->mMaxFingerNum)
    {
        TouchEvent currentTouchEvent;
        currentTouchEvent.fingerId = inFingerId;
        /* Wayland does not detect pressure */
        currentTouchEvent.pressure = 0;
        currentTouchEvent.xRelative = 0;
        currentTouchEvent.yRelative = 0;
        currentTouchEvent.eventType = TouchEvent::up;

        me->mCallbacks.onTouch(currentTouchEvent);
    }
}

void WaylandTouchListener::onMotion(void* inMe, wl_touch* inTouch, uint32_t inTimestamp, int32_t inFingerId,
        wl_fixed_t inPixelX, wl_fixed_t inPixelY)
{
    uspi_unused_variable(inTouch);
    uspi_unused_variable(inTimestamp);
    uspi_unused_variable(inFingerId);

    auto me = static_cast<WaylandTouchListener*>(inMe);

    me->mCallbacks.onLogging(USPI_LOG_VERBOSE, "Touch motion event finger: "
            + to_string(inFingerId) + "/" + to_string(me->mMaxFingerNum) + " on "
            + to_string(wl_fixed_to_double(inPixelX)) + " - " + to_string(wl_fixed_to_double(inPixelY)));

    if (inFingerId < me->mMaxFingerNum)
    {
        TouchEvent currentTouchEvent;
        currentTouchEvent.fingerId = inFingerId;
        /* Wayland does not detect pressure */
        currentTouchEvent.pressure = 1;
        currentTouchEvent.xRelative = (float)wl_fixed_to_double(inPixelX) / me->mDisplayWidth;
        currentTouchEvent.yRelative = (float)wl_fixed_to_double(inPixelY) / me->mDisplayHeight;
        currentTouchEvent.eventType = TouchEvent::move;

        me->mCallbacks.onTouch(currentTouchEvent);
    }
}

/* not interested in frame event */
void WaylandTouchListener::onFrame(void* inMe, wl_touch* inTouch)
{
    uspi_unused_variable(inMe);
    uspi_unused_variable(inTouch);
}

/* not interested in cancel event */
void WaylandTouchListener::onCancel(void* inMe, wl_touch* inTouch)
{
    uspi_unused_variable(inMe);
    uspi_unused_variable(inTouch);
}

} } /* namespace adit { namespace uspi { */
